import bpy
import os

# ------------------------------------------------------------
#   Property Groups
# ------------------------------------------------------------

class ExportCollectionEntry(bpy.types.PropertyGroup):
    """Properties for a single collection export entry."""
    collection: bpy.props.PointerProperty(
        name="Collection",
        type=bpy.types.Collection,
        description="Select a collection to export"
    )
    export_path: bpy.props.StringProperty(
        name="Export Path",
        subtype='DIR_PATH',
        default="//exported/",
        description="Path to export models from this collection"
    )


class ExportSettings(bpy.types.PropertyGroup):
    """Main export settings containing multiple collection entries."""
    export_entries: bpy.props.CollectionProperty(type=ExportCollectionEntry)

    export_format: bpy.props.EnumProperty(
        name="Format",
        items=[('FBX', "FBX", "Export to FBX format")],
        default='FBX',
        description="Choose the export file format"
    )

    show_exporter_settings: bpy.props.BoolProperty(
        name="Show Exporter Settings",
        default=True,
        description="Toggle visibility of the collection exporter settings"
    )

    # Persistent export settings
    apply_unit_scale: bpy.props.BoolProperty(default=True)

    fbx_axis_forward: bpy.props.EnumProperty(
        name="FBX Forward Axis",
        items=[
            ('X', "X", ""), ('Y', "Y", ""), ('Z', "Z", ""),
            ('-X', "-X", ""), ('-Y', "-Y", ""), ('-Z', "-Z", "")
        ],
        default='-Z'
    )
    fbx_axis_up: bpy.props.EnumProperty(
        name="FBX Up Axis",
        items=[
            ('X', "X", ""), ('Y', "Y", ""), ('Z', "Z", ""),
            ('-X', "-X", ""), ('-Y', "-Y", ""), ('-Z', "-Z", "")
        ],
        default='Y'
    )
    fbx_mesh_smooth_type: bpy.props.EnumProperty(
        name="FBX Smoothing",
        items=[('FACE', "Face", ""), ('EDGE', "Edge", ""), ('OFF', "Off", "")],
        default='FACE'
    )
    fbx_use_mesh_modifiers: bpy.props.BoolProperty(default=True)

    fbx_colors_type: bpy.props.EnumProperty(
        name="FBX Vertex Colors Type",
        items=[('NONE', "None", ""), ('SRGB', "sRGB", ""), ('LINEAR', "Linear", "")],
        default='LINEAR'
    )

    # Toggle for Auto-Baking logic
    fbx_prioritize_active_color: bpy.props.BoolProperty(
        name="Bake/Prioritize Vertex Color", 
        default=True,
        description="If enabled, converts mesh to vertex colors (baking diffuse) before export"
    )


# ------------------------------------------------------------
#   Export Settings Dialog
# ------------------------------------------------------------

class OBJECT_OT_open_export_settings(bpy.types.Operator):
    bl_idname = "export.open_export_settings"
    bl_label = "Export Settings"
    bl_description = "Open a window to configure advanced export settings"
    bl_options = {'REGISTER', 'UNDO'}

    apply_unit_scale: bpy.props.BoolProperty(default=True)

    fbx_axis_forward: bpy.props.EnumProperty(
        name="Forward Axis",
        items=[
            ('X', "X", ""), ('Y', "Y", ""), ('Z', "Z", ""),
            ('-X', "-X", ""), ('-Y', "-Y", ""), ('-Z', "-Z", "")
        ],
        default='-Z'
    )
    fbx_axis_up: bpy.props.EnumProperty(
        name="Up Axis",
        items=[
            ('X', "X", ""), ('Y', "Y", ""), ('Z', "Z", ""),
            ('-X', "-X", ""), ('-Y', "-Y", ""), ('-Z', "-Z", "")
        ],
        default='Y'
    )
    fbx_mesh_smooth_type: bpy.props.EnumProperty(
        name="Smoothing",
        items=[('FACE', "Face", ""), ('EDGE', "Edge", ""), ('OFF', "Off", "")],
        default='FACE'
    )
    fbx_use_mesh_modifiers: bpy.props.BoolProperty(default=True)

    fbx_colors_type: bpy.props.EnumProperty(
        name="Vertex Colors Type",
        items=[('NONE', "None", ""), ('SRGB', "sRGB", ""), ('LINEAR', "Linear", "")],
        default='LINEAR'
    )
    fbx_prioritize_active_color: bpy.props.BoolProperty(default=True)

    def invoke(self, context, event):
        s = context.scene.export_collection_settings
        self.apply_unit_scale = s.apply_unit_scale
        self.fbx_axis_forward = s.fbx_axis_forward
        self.fbx_axis_up = s.fbx_axis_up
        self.fbx_mesh_smooth_type = s.fbx_mesh_smooth_type
        self.fbx_use_mesh_modifiers = s.fbx_use_mesh_modifiers
        self.fbx_colors_type = s.fbx_colors_type
        self.fbx_prioritize_active_color = s.fbx_prioritize_active_color
        return context.window_manager.invoke_props_dialog(self)

    def draw(self, context):
        layout = self.layout
        settings = context.scene.export_collection_settings

        layout.label(text="Common Export Settings:")
        layout.prop(self, "apply_unit_scale")
        layout.separator()

        if settings.export_format == 'FBX':
            layout.label(text="FBX Specific Settings:")
            layout.prop(self, "fbx_axis_forward")
            layout.prop(self, "fbx_axis_up")
            layout.prop(self, "fbx_mesh_smooth_type")
            layout.prop(self, "fbx_use_mesh_modifiers")

            box_colors = layout.box()
            box_colors.label(text="Vertex Color / Baking:")
            box_colors.prop(self, "fbx_colors_type")
            box_colors.prop(self, "fbx_prioritize_active_color")

    def execute(self, context):
        s = context.scene.export_collection_settings
        s.apply_unit_scale = self.apply_unit_scale
        s.fbx_axis_forward = self.fbx_axis_forward
        s.fbx_axis_up = self.fbx_axis_up
        s.fbx_mesh_smooth_type = self.fbx_mesh_smooth_type
        s.fbx_use_mesh_modifiers = self.fbx_use_mesh_modifiers
        s.fbx_colors_type = self.fbx_colors_type
        s.fbx_prioritize_active_color = self.fbx_prioritize_active_color
        return {'FINISHED'}


# ------------------------------------------------------------
#   UI Operators for managing collection entries
# ------------------------------------------------------------

class OBJECT_OT_add_export_entry(bpy.types.Operator):
    bl_idname = "export.add_export_entry"
    bl_label = "Add Collection"
    bl_description = "Add a new collection export entry"

    def execute(self, context):
        context.scene.export_collection_settings.export_entries.add()
        return {'FINISHED'}


class OBJECT_OT_remove_export_entry(bpy.types.Operator):
    bl_idname = "export.remove_export_entry"
    bl_label = "Remove Collection"
    bl_description = "Remove this collection export entry"

    index: bpy.props.IntProperty()

    def execute(self, context):
        context.scene.export_collection_settings.export_entries.remove(self.index)
        return {'FINISHED'}


# ------------------------------------------------------------
#   Export Operator (main)
# ------------------------------------------------------------

class OBJECT_OT_export_collection_models(bpy.types.Operator):
    bl_idname = "export.collection_models"
    bl_label = "Export Selected Collections"
    bl_description = "Export models from all specified collections"

    _SUPPORTED_TYPES = {'MESH', 'CURVE', 'SURFACE', 'FONT'}

    # ---------- utilities ----------

    def _fbx_supports(self, prop_name: str) -> bool:
        try:
            return prop_name in bpy.ops.export_scene.fbx.get_rna_type().properties
        except Exception:
            return False

    def _ensure_object_mode(self):
        try:
            if bpy.context.mode != 'OBJECT':
                bpy.ops.object.mode_set(mode='OBJECT')
        except Exception:
            pass

    def _cleanup_objects(self, objs):
        """Remove objects safely."""
        for o in objs:
            if not o:
                continue
            try:
                bpy.data.objects.remove(o, do_unlink=True)
            except (ReferenceError, RuntimeError):
                pass

    def _ensure_temp_collection(self, name="__LPTK_EXPORT_TMP__"):
        scene_col = bpy.context.scene.collection
        tmp = bpy.data.collections.get(name)
        if tmp is None:
            tmp = bpy.data.collections.new(name)
            scene_col.children.link(tmp)
        return tmp

    def _remove_temp_collection_if_empty(self, tmp):
        try:
            if tmp and tmp.name in bpy.data.collections and len(tmp.objects) == 0:
                bpy.data.collections.remove(tmp)
        except Exception:
            pass

    def _iter_objects_recursive(self, col: bpy.types.Collection):
        """Yield objects in this collection and all sub-collections (recursive)."""
        for o in col.objects:
            yield o
        for child in col.children:
            yield from self._iter_objects_recursive(child)

    def _duplicate_link(self, obj: bpy.types.Object, tmp_col: bpy.types.Collection) -> bpy.types.Object:
        dup = obj.copy()
        if obj.data:
            dup.data = obj.data.copy()
        tmp_col.objects.link(dup)
        return dup

    def _convert_to_mesh(self, obj: bpy.types.Object) -> bool:
        self._ensure_object_mode()
        bpy.ops.object.select_all(action='DESELECT')
        obj.select_set(True)
        bpy.context.view_layer.objects.active = obj
        try:
            bpy.ops.object.convert(target='MESH')
            return True
        except RuntimeError:
            return False

    def _join_objects(self, objs):
        """Join objs; returns active object or None."""
        self._ensure_object_mode()
        if not objs:
            return None
        bpy.ops.object.select_all(action='DESELECT')
        
        # Ensure we have a valid active object for the join
        valid_objs = []
        for o in objs:
             try:
                 o.select_set(True)
                 valid_objs.append(o)
             except ReferenceError:
                 pass
        
        if not valid_objs:
            return None
            
        bpy.context.view_layer.objects.active = valid_objs[0]
        try:
            bpy.ops.object.join()
            return bpy.context.view_layer.objects.active
        except RuntimeError:
            return None

    # ---------- baking ----------

    def bake_diffuse_to_vertex_color(self, obj):
        """Bake diffuse into a VertexColor attribute called 'VertexColor'."""
        self._ensure_object_mode()

        mesh = obj.data
        color_name = "VertexColor"

        # Remove existing if present to ensure clean bake
        if color_name in mesh.color_attributes:
            try:
                mesh.color_attributes.remove(mesh.color_attributes[color_name])
            except Exception:
                pass

        vcol = mesh.color_attributes.new(name=color_name, type='BYTE_COLOR', domain='CORNER')
        mesh.color_attributes.active_color = vcol

        bpy.ops.object.select_all(action='DESELECT')
        obj.select_set(True)
        bpy.context.view_layer.objects.active = obj
        
        # Ensure render mode is set for display
        try:
            bpy.ops.geometry.color_attribute_render_set(name=color_name)
        except Exception:
            pass

        scene = bpy.context.scene
        prev_engine = scene.render.engine

        try:
            scene.render.engine = 'CYCLES'
        except Exception:
            print("Cycles not available; skipping bake.")
            return

        # Setup bake settings
        scene.cycles.bake_type = 'DIFFUSE'
        bake = scene.render.bake
        bake.target = 'VERTEX_COLORS'
        bake.use_pass_direct = False
        bake.use_pass_indirect = False
        bake.use_selected_to_active = False

        try:
            bpy.ops.object.bake(type='DIFFUSE')
        except Exception as e:
            print(f"Failed to bake diffuse for {obj.name}: {e}")
        finally:
            scene.render.engine = prev_engine

    # ---------- export helpers ----------

    def _export_fbx_selected(self, settings, filepath: str, no_materials: bool):
        fbx_kwargs = dict(
            filepath=filepath,
            use_selection=True,
            apply_unit_scale=settings.apply_unit_scale,
            axis_forward=settings.fbx_axis_forward,
            axis_up=settings.fbx_axis_up,
            object_types={'MESH'},
            mesh_smooth_type=settings.fbx_mesh_smooth_type,
            use_mesh_modifiers=settings.fbx_use_mesh_modifiers,
            bake_anim=False,
            bake_anim_simplify_factor=0.0,
            colors_type=settings.fbx_colors_type,
            prioritize_active_color=settings.fbx_prioritize_active_color,
        )
        # Handle older Blender versions or if property doesn't exist
        if no_materials and self._fbx_supports("use_materials"):
            fbx_kwargs["use_materials"] = False

        bpy.ops.export_scene.fbx(**fbx_kwargs)

    def _process_and_export_object(self, obj_to_export, export_name, export_path, settings) -> bool:
        """
        Final step: Bake (if enabled), clean materials (if baked), and export FBX.
        obj_to_export must be a duplicate/temp object.
        """
        bake_enabled = bool(settings.fbx_prioritize_active_color)
        
        if bake_enabled:
            self.bake_diffuse_to_vertex_color(obj_to_export)
            try:
                obj_to_export.data.materials.clear()
            except Exception:
                pass

        bpy.ops.object.select_all(action='DESELECT')
        obj_to_export.select_set(True)
        bpy.context.view_layer.objects.active = obj_to_export

        filepath = os.path.join(export_path, f"{export_name}.fbx")
        no_materials = bake_enabled

        try:
            self._export_fbx_selected(settings, filepath, no_materials=no_materials)
            return True
        except Exception as e:
            self.report({'ERROR'}, f"Failed to export {export_name}: {e}")
            return False

    def _export_child_collection_as_joined(self, child_col, export_path, settings, tmp_col, processed_objects_set) -> bool:
        """
        Gather ALL recursive objects in child_col, duplicate, join into ONE, export.
        Adds original objects to processed_objects_set.
        """
        # 1. Gather all valid source objects recursively
        sources = [o for o in self._iter_objects_recursive(child_col) if o.type in self._SUPPORTED_TYPES]
        
        if not sources:
            return False

        # Mark them as processed so the parent loop doesn't export them individually
        for s in sources:
            processed_objects_set.add(s)

        # 2. Duplicate them
        dups = []
        for src in sources:
            dup = self._duplicate_link(src, tmp_col)
            if self._convert_to_mesh(dup) and dup.type == 'MESH':
                dups.append(dup)
            else:
                self._cleanup_objects([dup])

        if not dups:
            return False

        # 3. Join
        if len(dups) == 1:
            joined = dups[0]
        else:
            joined = self._join_objects(dups)

        if not joined or joined.type != 'MESH':
            self._cleanup_objects(dups)
            return False

        # 4. Clean up leftovers from join
        leftovers = [o for o in dups if o != joined]
        self._cleanup_objects(leftovers)

        # 5. Export
        export_name = bpy.path.clean_name(child_col.name)
        joined.name = export_name
        
        success = self._process_and_export_object(joined, export_name, export_path, settings)
        
        # 6. Final cleanup of the joined temp object
        self._cleanup_objects([joined])
        
        return success

    def _export_single_object(self, src_obj, export_path, settings, tmp_col) -> bool:
        """Export one source object as one FBX file."""
        bpy.ops.object.select_all(action='DESELECT')

        dup = self._duplicate_link(src_obj, tmp_col)
        if not self._convert_to_mesh(dup) or dup.type != 'MESH':
            self._cleanup_objects([dup])
            return False

        export_name = bpy.path.clean_name(src_obj.name)
        success = self._process_and_export_object(dup, export_name, export_path, settings)
        
        self._cleanup_objects([dup])
        return success

    # ---------- main ----------

    def execute(self, context):
        settings = context.scene.export_collection_settings
        if not settings.export_entries:
            self.report({'ERROR'}, "No collections added for export.")
            return {'CANCELLED'}

        self._ensure_object_mode()

        # Save selection state
        prev_active = context.view_layer.objects.active
        prev_selected = [o for o in context.selected_objects]

        overall_success = True
        tmp_col = self._ensure_temp_collection()

        try:
            for entry in settings.export_entries:
                collection = entry.collection
                export_path = bpy.path.abspath(entry.export_path)

                if not collection:
                    continue

                os.makedirs(export_path, exist_ok=True)
                exported_count = 0
                
                # Keep track of objects handled by child collections
                processed_objects = set()

                # --- STEP 1: Export Child Collections (Joined) ---
                if collection.children:
                    for child in collection.children:
                        # This function bakes, joins, exports, AND updates processed_objects set
                        ok = self._export_child_collection_as_joined(child, export_path, settings, tmp_col, processed_objects)
                        if ok:
                            exported_count += 1
                        else:
                            # Not necessarily a failure, might be empty child col
                            pass

                # --- STEP 2: Export Objects directly in Parent (Single) ---
                # Only if they haven't been processed by the step above
                for obj in collection.objects:
                    if obj in processed_objects:
                        continue # Skip, already exported as part of a child group
                    
                    if obj.type not in self._SUPPORTED_TYPES:
                        continue

                    ok = self._export_single_object(obj, export_path, settings, tmp_col)
                    if ok:
                        exported_count += 1
                    else:
                        overall_success = False

                if exported_count > 0:
                    self.report({'INFO'}, f"Exported {exported_count} items from '{collection.name}'")
                else:
                    self.report({'WARNING'}, f"No models found in '{collection.name}'")

        finally:
            # Cleanup temp collection objects
            try:
                self._cleanup_objects(list(tmp_col.objects))
            except Exception:
                pass
            self._remove_temp_collection_if_empty(tmp_col)

            # Restore selection
            bpy.ops.object.select_all(action='DESELECT')
            for o in prev_selected:
                if o and o.name in bpy.data.objects:
                    try: o.select_set(True)
                    except: pass
            if prev_active and prev_active.name in bpy.data.objects:
                context.view_layer.objects.active = prev_active

        if overall_success:
            self.report({'INFO'}, "Export Process Completed.")
        else:
            self.report({'WARNING'}, "Export Completed with some errors.")
        return {'FINISHED'}


# ------------------------------------------------------------
#   Panel Drawing
# ------------------------------------------------------------

class ExporterPanelDrawing:
    def draw(self, context, layout):
        settings = context.scene.export_collection_settings
        box_exporter = layout.box()
        row = box_exporter.row()
        row.prop(
            settings, "show_exporter_settings",
            text="Collection Exporter",
            icon='EXPORT',
            emboss=False
        )

        if settings.show_exporter_settings:
            col = box_exporter.column(align=True)
            row_format = col.row(align=True)
            row_format.prop(settings, "export_format", text="Export Format")
            row_format.operator("export.open_export_settings", text="", icon='SETTINGS')
            col.separator()

            col.label(text="Collections to Export:")
            for i, entry in enumerate(settings.export_entries):
                entry_box = col.box()
                r = entry_box.row(align=True)
                r.prop(entry, "collection")
                remove_op = r.operator("export.remove_export_entry", text="", icon='TRASH')
                remove_op.index = i
                entry_box.prop(entry, "export_path")

            col.operator("export.add_export_entry", text="Add Collection", icon='ADD')
            col.separator()
            col.operator("export.collection_models", text="Start Export", icon='FILE_REFRESH')


# ------------------------------------------------------------
#   Registration
# ------------------------------------------------------------

# Helper to add panel to generic View3D > UI > Tool (or Misc)
class VIEW3D_PT_collection_exporter(bpy.types.Panel):
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = 'Exporter'
    bl_label = "Batch Exporter"

    def draw(self, context):
        ExporterPanelDrawing().draw(context, self.layout)

exporter_classes = [
    ExportCollectionEntry,
    ExportSettings,
    OBJECT_OT_add_export_entry,
    OBJECT_OT_remove_export_entry,
    OBJECT_OT_open_export_settings,
    OBJECT_OT_export_collection_models,
    VIEW3D_PT_collection_exporter
]

def register():
    for cls in exporter_classes:
        bpy.utils.register_class(cls)
    bpy.types.Scene.export_collection_settings = bpy.props.PointerProperty(type=ExportSettings)

def unregister():
    del bpy.types.Scene.export_collection_settings
    for cls in reversed(exporter_classes):
        bpy.utils.unregister_class(cls)

if __name__ == "__main__":
    register()